<?php 
namespace App\Http\Controllers\Admin;

use App\Models\Movie;
use Illuminate\Support\Facades\Input;
use Illuminate\Http\Request;
use Validator;
use Config;

use Illuminate\Routing\Controller as BaseController;

class MovieController extends BaseController
{
    public function massEdit()
    {
        return view('admin.movie.mass_edit')->with('title','Mass Edit');
    }
        
    public function massUpdate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'keyword' => 'required|max:100',
            'links' => 'required'
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                        ->withErrors($validator)
                        ->withInput();
        }
        $slugs = [];

        $old_items = [];
        foreach(preg_split("/((\r?\n)|(\r\n?))/", $request->links) as $link){
            $link = explode('/',$link);
            $slug = end($link);
                            
            $pattern = '~\b\d{4}\b\+?~';
            preg_match($pattern, $slug,$match);
            
            if(!empty($match[0]))
            {
                $pos = strpos($slug,$match[0]);
                $slug = substr($slug,0,$pos+4);
            }  
            $slug = trim($slug);
            array_push($slugs,$slug);
        } 
        
        $slugs = array_unique($slugs);
        $updated = [];
        foreach($slugs as $slug)
        {
            $movie = Movie::where('slug','like',$slug.'%')->first();
            if(!empty($movie))
            {
                $old_slug = $movie->slug;
                $movie->slug = str_slug($movie->title_long.' '.$request->keyword);
                $movie->findme = $request->keyword;
                $movie->save();
                
                array_push($old_items,["movie_id"=>$movie->id, "slug"=>$old_slug]);
                array_push($updated,$slug);
            }
        }
        if(!empty($old_items)) \App\Models\OldLink::insert($old_items);
        
        $info = '';
        if(count($slugs) != count($updated)){ 
            $diff_slugs = array_diff($slugs,$updated);
            $info = ' For some reasons this movies were not updated - '.implode(', ',$diff_slugs);
            $log = [];
            foreach($diff_slugs as $diff_slug)
            {
                array_push($log,route('movie.show',[$diff_slug]));
            }
            session(['info'=>implode('\r\n ',$log)]);
        }
        
        return redirect()->back()->withSuccess(count($updated).' Movies found and updated from given '.count($slugs).' links.'.$info);
    }
    
	public function create()
	{
		return view('admin.movie.create');
	}
	
	public function save(Request $request)
	{
        $validator = Validator::make($request->all(), [
            'title' => 'required|max:100',
            'title_long' => 'required|max:120|unique:movie,title_long',
            'imdb_code' => 'nullable|max:120|unique:movie,imdb_code',
            'year' => 'required|numeric',
            'rating' => 'required|numeric|min:1|max:10',
            'runtime' => 'required|numeric',
            'genres' => 'required',
            'cast' => 'nullable',
            'active' => 'required|in:0,1',
            'director' => 'nullable|max:200',
            'description_full' => 'required',
            'yt_trailer_code' => 'nullable',
            'language' => 'required',
            'mpa_rating' => 'nullable',
            'tomato_critics' => 'required|min:1|max:100',
            'tomato_audience' => 'required|min:1|max:100',
            'cover_image' => 'required|mimes:jpg,jpeg,png|max:'.config('settings.max_screenshot_size'),            
            'screenshots.*' => 'nullable|mimes:jpg,jpeg,png|max:'.config('settings.max_screenshot_size'),
            'background_image' => 'nullable|mimes:jpg,jpeg,png|max:'.config('settings.max_screenshot_size'),
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                        ->withErrors($validator)
                        ->withInput();
        }
        else {   
        	$movie = new Movie();
        	$movie->title = $request->title;
        	$movie->title_long = $request->title_long;
        	$movie->imdb_code = $request->imdb_code;
        	$movie->year = $request->year;
        	$movie->rating = $request->rating;
        	$movie->runtime = $request->runtime;
        	$movie->genres = implode(',', $request->genres);
        	$movie->cast = $request->cast;
        	$movie->director = $request->director;
        	$movie->description_full = $request->description_full;
        	$movie->yt_trailer_code = $request->yt_trailer_code;
        	$movie->language = $request->language;
        	$movie->mpa_rating = $request->mpa_rating;
        	$movie->tomato_critics = $request->tomato_critics;
        	$movie->tomato_audience = $request->tomato_audience;
        	$movie->active = $request->active;
        	$movie->direct_link = $request->direct_link;
        	$movie->findme = $request->findme;

            $destinationPath = 'uploads/movies/'.str_slug($movie->title_long.' '.$movie->imdb_code);
            if (!file_exists($destinationPath)) {
				mkdir($destinationPath, 0755, true);
				chmod($destinationPath, 0755);       
			}

        	$cover_image = $request->file('cover_image');
            $file_ext = $cover_image->getClientOriginalExtension();				     
            $new_file_name = "cover-original.".$file_ext;
			$cover_image->move($destinationPath,$new_file_name);    
			$original_cover_image = $destinationPath."/".$new_file_name;


			$img = \Image::make($original_cover_image);
			$img->resize(230, 345);
			$resized_cover_image = $destinationPath."/"."cover-medium.".$file_ext;
			$img->save($resized_cover_image);			 
			$movie->cover_image = "/".$resized_cover_image;
			

			if ($request->hasFile('background_image')) {

        		$background_image = $request->file('background_image');					
				$file_ext = $background_image->getClientOriginalExtension();	
				$new_file_name = "background.".$file_ext;
				$background_image->move($destinationPath,$new_file_name); 
				$movie->background_image = "/".$destinationPath."/".$new_file_name;   
			}

			if($request->hasFile('screenshots')){
				$screenshots = [];
				$i = 1;
				foreach($request->file('screenshots') as $screenshot) {
					$file_ext = $screenshot->getClientOriginalExtension();	
					$new_file_name = "screenshot-".$i.".".$file_ext;
					$screenshot->move($destinationPath,$new_file_name); 
					array_push($screenshots, "/".$destinationPath."/".$new_file_name);
					$i++;									
				}
				$movie->screenshots = json_encode(array_values($screenshots));
			}


			$movie->save();
			return redirect()->back()->withSuccess('Movie successfully added.');
        }


	}

	public function edit($id)
	{
		$movie = Movie::findOrfail($id);
		return view('admin.movie.edit',compact('movie'));
	}	

	public function update($id,Request $request)
	{
        $validator = Validator::make($request->all(), [
            'title' => 'required|max:100',
            'title_long' => 'required|max:120|unique:movie,title_long,'.$id,
            'slug' => 'required|max:255|unique:movie,title_long,'.$id,
            'imdb_code' => 'nullable|max:120|unique:movie,imdb_code,'.$id,
            'year' => 'required|numeric',
            'rating' => 'required|numeric|min:1|max:10',
            'runtime' => 'required|numeric',
            'genres' => 'required',
            'cast' => 'nullable',
            'active' => 'required|in:0,1',
            'featured' => 'required|in:0,1',
            'director' => 'nullable',
            'description_full' => 'required',
            'yt_trailer_code' => 'nullable',
            'language' => 'required',
            'mpa_rating' => 'nullable',
            'tomato_critics' => 'required|min:1|max:100',
            'tomato_audience' => 'required|min:1|max:100',
            'cover_image' => 'nullable|mimes:jpg,jpeg,png|max:'.config('settings.max_screenshot_size'),            
            'screenshots.*' => 'nullable|mimes:jpg,jpeg,png|max:'.config('settings.max_screenshot_size'),
            'background_image' => 'nullable|mimes:jpg,jpeg,png|max:'.config('settings.max_screenshot_size'),
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                        ->withErrors($validator)
                        ->withInput();
        }
        else {   
        	$movie = Movie::findOrfail($id);
        	$old_slug = $movie->slug;
        	
        	$movie->title = $request->title;
        	$movie->title_long = $request->title_long;
        	$movie->slug = $request->slug;
        	$movie->imdb_code = $request->imdb_code;
        	$movie->year = $request->year;
        	$movie->rating = $request->rating;
        	$movie->runtime = $request->runtime;
        	$movie->genres = implode(',', $request->genres);
        	$movie->cast = $request->cast;
        	$movie->director = $request->director;
        	$movie->description_full = $request->description_full;
        	$movie->yt_trailer_code = $request->yt_trailer_code;
        	$movie->language = $request->language;
        	$movie->mpa_rating = $request->mpa_rating;
        	$movie->tomato_critics = $request->tomato_critics;
        	$movie->tomato_audience = $request->tomato_audience;
        	$movie->active = $request->active;
        	$movie->direct_link = $request->direct_link;
        	$movie->created_at = $request->created_at;
        	$movie->findme = $request->findme;
        	$movie->featured = $request->featured;

            $destinationPath = 'uploads/movies/'.str_slug($movie->title_long.' '.$movie->imdb_code);
            if (!file_exists($destinationPath)) {
				mkdir($destinationPath, 0755, true);
				chmod($destinationPath, 0755);       
			}

			if ($request->hasFile('cover_image')) {
	        	$cover_image = $request->file('cover_image');
	            $file_ext = $cover_image->getClientOriginalExtension();				     
	            $new_file_name = "cover-original.".$file_ext;
				$cover_image->move($destinationPath,$new_file_name);    
				$original_cover_image = $destinationPath."/".$new_file_name;


				$img = \Image::make($original_cover_image);
				$img->resize(230, 345);
				$resized_cover_image = $destinationPath."/"."cover-medium.".$file_ext;
				$img->save($resized_cover_image);			 
				$movie->cover_image = "/".$resized_cover_image;
			}

			if ($request->hasFile('background_image')) {
        		$background_image = $request->file('background_image');					
				$file_ext = $background_image->getClientOriginalExtension();	
				$new_file_name = "background.".$file_ext;
				$background_image->move($destinationPath,$new_file_name); 
				$movie->background_image = "/".$destinationPath."/".$new_file_name;   
			}

			if($request->hasFile('screenshots')){
				$screenshots = array();
				$i = 1;
				foreach($request->file('screenshots') as $screenshot) {
					$file_ext = $screenshot->getClientOriginalExtension();	
					$new_file_name = "screenshot-".$i.".".$file_ext;
					$screenshot->move($destinationPath,$new_file_name); 
					$random = rand(10000,99999);
					array_push($screenshots, "/".$destinationPath."/".$new_file_name.'?t='.$random);
					$i++;									
				}
				$movie->screenshots = json_encode(array_values($screenshots));
			}


			$movie->save();
			
			
			if($old_slug != $request->slug) \App\Models\OldLink::insert(['movie_id'=>$movie->id,'slug'=>$old_slug]);
			return redirect()->back()->withSuccess('Movie successfully updated');
        }

	}

    public function updateSelected(Request $request)
    {
        $ids = $request->ids;
        $count = count($ids);
        if($count > 0)
        {
            $old_items = [];
            foreach($ids as $id){
              $item = Movie::find($id);
              if(!empty($item))
              {
                $old_slug = $item->slug;
                  
                  
                if(!empty($request->replace))
                {
                      if(!empty($item->findme) && !empty($request->find)) $item->slug = str_replace($request->find,$request->replace,$item->slug); 
                      else $item->slug = $item->slug.'-'.$request->replace;
                      $item->findme = $request->replace;
                } 
                  
                  if(!empty($request->created_at)) $item->created_at = $request->created_at;
                  
                  if($request->featured > 0)
                  {
                      $item->featured = ($request->featured == 1) ? 1 : 0;
                  }
                  $item->save();
                  
                  
                  array_push($old_items,["movie_id"=>$item->id, "slug"=>$old_slug]);
                  
              }
              
            }
            if(!empty($old_items)) \App\Models\OldLink::insert($old_items);
            return '<div class="alert alert-success">'. $count.' Movies successfully updated</div>';
            
        }
        else{
            return '<div class="alert alert-danger">No Movies selected</div>';
        }
    }  
    
    public function autoUpdateSlugs(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'genre' => 'required|string|exists:genre,name',
            'crt_type' => 'required|string|in:add,sub,now',
            'crt_number' => 'required|numeric',
            'crt' => 'required|string|in:months,days,hours,minutes,seconds',
        ]);
        if ($validator->fails()) {
            return $validator->errors()->all();
        }        
      
        if($request->crt_type == "add")
        {
            switch ($request->crt) {
                case 'months':
                    $created_at = \Carbon\Carbon::now()->addMonths($request->crt_number)->format('Y-m-d H:i:s');
                    break;                
                
                case 'days':
                    $created_at = \Carbon\Carbon::now()->addDays($request->crt_number)->format('Y-m-d H:i:s');
                    break;                
                
                case 'hours':
                    $created_at = \Carbon\Carbon::now()->addHours($request->crt_number)->format('Y-m-d H:i:s');
                    break;                
                
                case 'minutes':
                    $created_at = \Carbon\Carbon::now()->addMinutes($request->crt_number)->format('Y-m-d H:i:s');
                    break;                
                
                case 'seconds':
                    $created_at = \Carbon\Carbon::now()->addSeconds($request->crt_number)->format('Y-m-d H:i:s');
                    break;
                
                default:
                    $created_at = \Carbon\Carbon::now()->format('Y-m-d H:i:s');
                    break;
            }
            
        }
        elseif($request->crt_type == "sub")
        {
            switch ($request->crt) {
                case 'months':
                    $created_at = \Carbon\Carbon::now()->subMonths($request->crt_number)->format('Y-m-d H:i:s');
                    break;                
                
                case 'days':
                    $created_at = \Carbon\Carbon::now()->subDays($request->crt_number)->format('Y-m-d H:i:s');
                    break;                
                
                case 'hours':
                    $created_at = \Carbon\Carbon::now()->subHours($request->crt_number)->format('Y-m-d H:i:s');
                    break;                
                
                case 'minutes':
                    $created_at = \Carbon\Carbon::now()->subMinutes($request->crt_number)->format('Y-m-d H:i:s');
                    break;                
                
                case 'seconds':
                    $created_at = \Carbon\Carbon::now()->subSeconds($request->crt_number)->format('Y-m-d H:i:s');
                    break;
                
                default:
                    $created_at = \Carbon\Carbon::now()->format('Y-m-d H:i:s');
                    break;
            }
        }
        else{
            $created_at = \Carbon\Carbon::now()->format('Y-m-d H:i:s');
        }
        
      
        $movies = \App\Models\Movie::where('genres','like','%'.$request->genre.'%')->get(['id']);
        
        $random_word = substr(str_shuffle("abcdefghijklmnopqrstvwxyz"), 0, 3) . 'torrent';
        $random_number = rand(1,9) . "00";
        $random_replace = $random_word.$random_number;
        $old_items = [];
        foreach($movies as $m)
        {

              $item = \App\Models\Movie::find($m->id);
              if(!empty($item))
              {
                $old_slug = $item->slug;
                  
                if(!empty($item->findme)) $item->slug = str_replace($item->findme,$random_replace,$item->slug); 
                else $item->slug = $item->slug.'-'.$random_replace;
                $item->findme = $random_replace;
                 
                  
                $item->created_at = ($item->featured == 1) ? \Carbon\Carbon::parse($created_at)->addMinutes(1)->format('Y-m-d H:i:s') : $created_at;
                $item->save();
                  
                  
                array_push($old_items,["movie_id"=>$item->id, "slug"=>$old_slug]);
                  
              }
              
        }
        if(!empty($old_items)) \App\Models\OldLink::insert($old_items);
        return count($old_items).' Movies successfully updated';          
    }
    
    
}