<?php 
namespace App\Http\Controllers\Admin;

use Backpack\CRUD\app\Http\Controllers\CrudController;

// VALIDATION: change the requests to match your own file names if you need form validation
//use App\Http\Requests\MovieCrudRequest as StoreRequest;
//use App\Http\Requests\MovieCrudRequest as UpdateRequest;

use App\Models\Genre;
use App\Models\Movie;
use App\Models\Torrent;
use App\Models\Video;
use App\Models\Subtitle;

use Illuminate\Http\Request;
use Symfony\Component\DomCrawler\Crawler;

class MovieCrudController extends CrudController {

	public function setup() 
	{
        $this->crud->setModel("App\Models\Movie");
        $this->crud->setRoute("admin/movie");
        $this->crud->setEntityNameStrings('movie', 'movies');

        $this->crud->setColumns([['name'=>'id','label'=>'#','type'=>'mcheck'],'title_long','slug','findme','imdb_code','year','rating','created_at','active','featured']);        

        $this->crud->enableAjaxTable(['title_long','slug','findme','genre','imdb_code','year','rating']);
        $this->crud->removeButton('create');
        $this->crud->removeButton('update');
        $this->crud->addButtonFromView('top', 'add_movie', 'add', 'end'); 
        $this->crud->addButtonFromView('line', 'edit_movie', 'edit', 'beginning'); 
        $this->crud->addButtonFromView('line', 'torrent', 'torrent', 'beginning'); 
        $this->crud->addButtonFromView('line', 'video', 'video', 'beginning'); 
      //  $this->crud->addButtonFromModelFunction('line', 'open_google', 'openGoogle', 'beginning'); 
        $this->crud->addButtonFromView('line', 'open_movie', 'open_movie', 'beginning'); 
        $this->crud->enableDetailsRow();
        $this->crud->allowAccess('details_row');

        $this->crud->addButtonFromView('top', 'select_all', 'select_all', 'end');         
        $this->crud->addButtonFromView('top', 'edit_selected', 'edit_selected', 'end');          
        $this->crud->addButtonFromModelFunction('line', 'get_tgx', 'getTgX', 'beginning');
        $this->crud->addButtonFromModelFunction('line', 'get_tmdb_images', 'getTmdbImages', 'beginning');
    
        $this->crud->addFilter([ // dropdown filter
          'name' => 'mpa_rating',
          'type' => 'dropdown',
          'label'=> 'MPA Rating'
        ], [
          'G' => 'G',
          'PG' => 'PG',
          'PG-13' => 'PG-13',
          'R' => 'R',
          'NC-17' => 'NC-17',
        ], function($value) { // if the filter is active
             $this->crud->addClause('where', 'mpa_rating',$value);
        });   

        $this->crud->addFilter([ // select2_multiple filter
          'name' => 'genres',
          'type' => 'select2',
          'label'=> 'Genres'
        ], function() { // the options that show up in the select2
            $genres = Genre::all()->pluck('name')->toArray();
            $gen = array();
            foreach($genres as $genre){
                $gen[$genre] = $genre;
            }
            return $gen;
        }, function($value) { // if the filter is active
                $this->crud->addClause('where','genres','like','%'.$value.'%');
        }); 

        $this->crud->addFilter([ // select2_multiple filter
          'name' => 'genres_not',
          'type' => 'select2',
          'label'=> 'Genres Not'
        ], function() { // the options that show up in the select2
            $genres = Genre::all()->pluck('name')->toArray();
            $gen = array();
            foreach($genres as $genre){
                $gen[$genre] = $genre;
            }
            return $gen;
        }, function($value) { // if the filter is active
                $this->crud->addClause('where','genres','not like','%'.$value.'%');
        });  
        
        
        $this->crud->addFilter([ // dropdown filter
          'name' => 'torrents',
          'type' => 'dropdown',
          'label'=> 'Torrents',
          'placeholder' => 'Torrents'
        ], [
          '1' => 'Has',
          '2' => 'Has not',
        ], function($value) { // if the filter is active
            if($value == '1') $this->crud->addClause('whereHas', 'torrents');
            else $this->crud->addClause('doesntHave', 'torrents');
        });         

        $this->crud->addFilter([ // select2_ajax filter
          'name' => 'Artist',
          'type' => 'select2_ajax',
          'label'=> 'Cast',
          'placeholder' => 'Artist name'
        ],
        url('admin/artist/ajax-artist-names'), // the ajax route
        function($value) { // if the filter is active
             $this->crud->addClause('where', 'cast','like','%'. $value .'%');
        });

        $this->crud->addFilter([ // dropdown filter
          'name' => 'screenshots',
          'type' => 'dropdown',
          'label'=> 'Screenshots',
          'placeholder' => 'Screenshots'
        ], [
          '1' => 'Has',
          '2' => 'Has not',
        ], function($value) { // if the filter is active
            if($value == '1') $this->crud->addClause('where', 'screenshots','!=','');
            else $this->crud->addClause('where', 'screenshots','=',null);
        });         

        $this->crud->addFilter([ // dropdown filter
          'name' => 'background',
          'type' => 'dropdown',
          'label'=> 'Background',
          'placeholder' => 'Background Image'
        ], [
          '1' => 'Has',
          '2' => 'Has not',
        ], function($value) { // if the filter is active
            if($value == '1') $this->crud->addClause('where', 'background_image','!=','');
            else $this->crud->addClause('where', 'background_image','=',null);
        });         
        
        $this->crud->addFilter([ // dropdown filter
          'name' => 'featured',
          'type' => 'dropdown',
          'label'=> 'Featured',
          'placeholder' => 'Featured'
        ], [
          1 => 'Yes',
          2 => 'No',
        ], function($value) { // if the filter is active
            if($value == 1) $this->crud->addClause('where', 'featured',1);
            else $this->crud->addClause('where', 'featured',0);
        }); 

        $this->crud->addFilter([ // dropdown filter
          'name' => 'status',
          'type' => 'dropdown',
          'label'=> 'Rating'
        ], [
          '10' => '10+',
          '9' => '9+',
          '8' => '8+',
          '7' => '7+',
          '6' => '6+',
          '5' => '5+',
          '4' => '4+',
          '3' => '3+',
          '2' => '2+',
          '1' => '1+',
        ], function($value) { // if the filter is active
             $this->crud->addClause('where', 'rating','>=',$value);
        });  
        
        $this->crud->addFilter([ // simple filter
          'type' => 'text',
          'name' => 'findme',
          'label'=> 'Findme'
        ], 
        false, 
        function($value) { // if the filter is active
             $this->crud->addClause('where', 'findme',$value);
        }); 
        
        $this->crud->addFilter([ // dropdown filter
          'name' => 'order_by',
          'type' => 'dropdown',
          'label'=> 'Order By'
        ], [
          'new' => 'Newly added',
          'name' => 'Name',
          'best' => 'Rating',
          'year' => 'Year',
          'download_count' => 'Download Count',
        ], function($value) { // if the filter is active
             if($value == 'new') $this->crud->orderBy('created_at','DESC');
             elseif($value == 'title_long') $this->crud->orderBy('title_long');
             elseif($value == 'best') $this->crud->orderBy('like_count','DESC');
             elseif($value == 'year') $this->crud->orderBy('year','DESC');
             elseif($value == 'download_count') $this->crud->orderBy('download_count','DESC');
             else $this->crud->orderBy('id','DESC');
        });   
        
        if(empty(request()->get('order'))) $this->crud->orderBy('created_at','DESC');
        
    }   

	public function store(StoreRequest $request)
	{
		return parent::storeCrud();
	}

	public function update(UpdateRequest $request)
	{
		return parent::updateCrud();
	}

    public function destroy($id)
    {
      $movie = Movie::findOrfail($id);
      $imageFolderPath = 'images/movies/'.str_slug($movie->title_long.' '.$movie->imdb_code);
      $movieFolderPath = 'uploads/movies/'.str_slug($movie->title_long.' '.$movie->imdb_code);
      $subtitleFolderPath = 'uploads/movies/'.str_slug($movie->title_long.' '.$movie->imdb_code).'/subtitles';
      if(file_exists($imageFolderPath)){        
        chmod($imageFolderPath, 0777);  
        $this->remove_directory($imageFolderPath);
      }      
      if(file_exists($movieFolderPath)){
        chmod($movieFolderPath, 0777);  
        $this->remove_directory($movieFolderPath);
      }      

      return $this->crud->delete($id);
    } 

    public function showDetailsRow($id)
    {
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view('crud::details_row.movie', $this->data);
    } 

    protected static function remove_directory($dir)
    {
      $di = new \RecursiveDirectoryIterator($dir, \FilesystemIterator::SKIP_DOTS);
      $ri = new \RecursiveIteratorIterator($di, \RecursiveIteratorIterator::CHILD_FIRST);
      foreach ( $ri as $file ) {
          $file->isDir() ?  rmdir($file) : unlink($file);
      }
      if(file_exists($dir)) rmdir($dir);
      return true;      
    }
    
    public function tgx($id)
    {
        $movie = Movie::findOrfail($id);
        return view('admin.movie.tgx',compact('movie'))->with('title','TGX - '.$movie->title_long);
    }
    
    public function tgxStore(Request $request)
    {
        config(['app.debug'=>true]);
        $validator = \Validator::make($request->all(), [
            'movie_id' => 'required|numeric|exists:movie,id',
            'page_link' => 'required|url'
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                        ->withErrors($validator)
                        ->withInput();
        }  
        
        try{
            $html = $this->curl_get_contents($request->page_link);
        }
        catch(\Exception $e){
            return 'api error';
        }
        
        $movie = \App\Models\Movie::where('id',$request->movie_id)->firstOrfail();

        $crawler = new Crawler($html);
        
        $torrent_name = $crawler->filter('span.linebreakup')->eq(0)->text();
        
        $hash = $crawler->filter('span.linebreakup')->eq(1)->text();
        
        $magnet = $crawler->filter('a.btn-danger')->eq(0)->attr('href');
        
        $torrent_url = $crawler->filter('a.btn-success')->attr('href');
        
        $seeds = ($crawler->filter('font[color="green"]')->count() > 0) ? $crawler->filter('font[color="green"]')->text() : 0;
        $peers = ($crawler->filter('font[color="#ff0000"]')->count() > 0) ? $crawler->filter('font[color="#ff0000"]')->text() : 0;
        
        $torrent_size = 0;

        $torrent_size = $crawler->filter('div.limitwidth > div.tprow')->each(function($node){
 
            if(strpos($node->filter('div')->text(),'Size:') > 0)
            {
                $torrent_size = str_replace('Total Size:','',$node->filter('div')->text());
                $torrent_size = str_replace(',','',$torrent_size);
                $torrent_size = $this->toByteSize(trim($torrent_size));
                return $torrent_size;
            }
        });
        $torrent_size = (!empty(array_values(array_filter($torrent_size))[0])) ? array_values(array_filter($torrent_size))[0] : 0;
        $torrent_size = str_replace(',','',$torrent_size);
        

        $torrent = \App\Models\Torrent::where('hash',$hash)->where('movie_id',$movie->id)->first();
        if(empty($torrent))
        { 

            $torrent = new \App\Models\Torrent();
            $torrent->movie_id = $movie->id;
            $torrent->name = $torrent_name;
            $torrent->hash = $hash;
            $torrent->quality = $this->getQuality($torrent_name);
            $torrent->resolution = $this->getResolution($torrent->quality);
            $torrent->size_bytes = $torrent_size; 
            $torrent->magnet = $magnet; 

            $torrentPath = 'uploads/movies/'.str_slug($movie->title_long);
            if (!file_exists($torrentPath)) {
                mkdir($torrentPath, 0775, true);
                chmod($torrentPath, 0775);       
            }                    
            $torrent_file = $torrentPath."/".str_slug($movie->title_long)."-".str_slug($torrent->quality)."-".str_random(3).".torrent";
            try{    
             //   file_put_contents($torrent_file, fopen($torrent_url, 'r'));  
                $torrent_file = "/".$torrent_file;  
            }   
            catch(\Exception $e)    
            {   
                $torrent_file = $magnet; 
            } 

            $torrent->file = $torrent_file;      

            $movie->created_at = date('Y-m-d H:i:s');             
            $movie->save();
            
            $status = "added";

        }
        else{
            $status = "updated";
        }

        $torrent->seeds = $seeds; 
        $torrent->peers = $peers; 
        $torrent->save();
      
        return redirect()->back()->withSuccess('Successfully '.$status)->withInput();
    }
    
    public function t1337Store(Request $request)
    {
        config(['app.debug'=>true]);
        $validator = \Validator::make($request->all(), [
            'movie_id' => 'required|numeric|exists:movie,id',
            'torrent_link' => 'required|url'
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                        ->withErrors($validator)
                        ->withInput();
        }  
        
        try{
            $html = $this->curl_get_contents($request->torrent_link);
        }
        catch(\Exception $e){
            return 'api error';
        }
        
        $movie = \App\Models\Movie::where('id',$request->movie_id)->firstOrfail();

        $crawler = new Crawler($html);
        
        $torrent_name = $crawler->filter('.box-info-heading')->text();
        $torrent_name = trim($torrent_name);
        
        $hash = $crawler->filter('.infohash-box > p > span')->text();
        
        $magnet = $crawler->filter('.box-info > div')->eq(1)->filter('div > ul > li > a')->attr('href');
        
        $torrent_url = $crawler->filter('.box-info > div')->eq(1)->filter('div > ul > li.dropdown')->filter('ul > li > a')->attr('href');
        
        $seeds = ($crawler->filter('.seeds')->count() > 0) ? $crawler->filter('.seeds')->text() : 0;
        $peers = ($crawler->filter('.leeches')->count() > 0) ? $crawler->filter('.leeches')->text() : 0;
        
        $torrent_size = 0;

        $torrent_size = $crawler->filter('ul.list')->eq(1)->filter('li')->each(function($node){
          
            if(strpos($node->filter('strong')->text(),'size') > 0)
            {
                $torrent_size = str_replace(',','',$node->filter('span')->text());
                $torrent_size = $this->toByteSize(trim($torrent_size));
                   
                return $torrent_size;
                
            }
        });

        $torrent_size = (!empty(array_values(array_filter($torrent_size))[0])) ? array_values(array_filter($torrent_size))[0] : 0;
        $torrent_size = str_replace(',','',$torrent_size);
        

        $torrent = \App\Models\Torrent::where('hash',$hash)->where('movie_id',$movie->id)->first();
        if(empty($torrent))
        { 

            $torrent = new \App\Models\Torrent();
            $torrent->movie_id = $movie->id;
            $torrent->name = $torrent_name;
            $torrent->hash = $hash;
            $torrent->quality = $this->getQuality($torrent_name);
            $torrent->resolution = $this->getResolution($torrent->quality);
            $torrent->size_bytes = $torrent_size; 
            $torrent->magnet = $magnet; 

            $torrentPath = 'uploads/movies/'.str_slug($movie->title_long);
            if (!file_exists($torrentPath)) {
                mkdir($torrentPath, 0775, true);
                chmod($torrentPath, 0775);       
            }                    
            $torrent_file = $torrentPath."/".str_slug($movie->title_long)."-".str_slug($torrent->quality)."-".str_random(3).".torrent";
            try{    
             //   file_put_contents($torrent_file, fopen($torrent_url, 'r'));  
                $torrent_file = "/".$torrent_file;  
            }   
            catch(\Exception $e)    
            {   
                $torrent_file = $magnet; 
            } 

            $torrent->file = $torrent_file;      

            $movie->created_at = date('Y-m-d H:i:s');             
            $movie->save();
            
            $status = "added";

        }
        else{
            $status = "updated";
        }

        $torrent->seeds = $seeds; 
        $torrent->peers = $peers; 
        $torrent->save();
      
        return redirect()->back()->withSuccess('Successfully '.$status)->withInput();
    }    
    protected static function toByteSize($size) 
    {
        if(strpos($size,'KB') > 0)
        {
          $size = trim(str_replace('KB','',$size));
          return $fileSize = round($size * 1000);	
        }
        elseif(strpos($size,'MB') > 0)
        {
            $size = trim(str_replace('MB','',$size));
            return $fileSize = round($size * 1000 * 1000);	
        }
        elseif(strpos($size,'GB') > 0)
        {
            $size = trim(str_replace('GB','',$size));
            return $fileSize = round($size * 1000 * 1000 * 1000);	
        }
        else
        {
            return 0;
        }
    }

    protected static function getQuality($name)
    {
        if (strpos($name, '420p') !== false) {
            $quality = '420p';
        }
        elseif (strpos($name, 'HDCAM') !== false) {
            $quality = 'HDCAM';
        }        
        elseif (strpos($name, 'HDTS') !== false) {
            $quality = 'HDTS';
        }        
        elseif (strpos($name, '720p') !== false) {
            $quality = '720p';
        }
        elseif (strpos($name, '1080p') !== false) {
            $quality = '1080p';
        }         
        elseif (strpos($name, '2160p') !== false) {
            $quality = '2160p';
        }        
        elseif (strpos($name, 'blueray') !== false) {
            $quality = 'blueray';
        }        
        elseif (strpos($name, '3D') !== false) {
            $quality = '3D';
        }        
        else{
            $quality = '720p';
        }
        return $quality;
    }    

    protected static function getResolution($torrent_quality)
    {
        if($torrent_quality == '3D'){
            $torrent_resolution = '1920*1080';
        }
        elseif($torrent_quality == '2160p'){
            $torrent_resolution = '3840*2160';
        }                            
        elseif($torrent_quality == '1080p'){
            $torrent_resolution = '1920*1080';
        }
        elseif ($torrent_quality == '720p') {
            $torrent_resolution = '1280*720';
        }
        else{
            $torrent_resolution = '1280*720';
        }
        return $torrent_resolution;
    }
    
    protected static function curl_get_contents($url)
    {
        $user_agent='Mozilla/5.0 (Windows NT 6.1; rv:8.0) Gecko/20100101 Firefox/8.0';

        $options = array(

            CURLOPT_CUSTOMREQUEST  =>"GET",        //set request type post or get
            CURLOPT_POST           =>false,        //set to GET
            CURLOPT_USERAGENT      => $user_agent, //set user agent
            CURLOPT_COOKIEFILE     =>"cookie.txt", //set cookie file
            CURLOPT_COOKIEJAR      =>"cookie.txt", //set cookie jar
            CURLOPT_RETURNTRANSFER => true,     // return web page
            CURLOPT_HEADER         => false,    // don't return headers
            CURLOPT_FOLLOWLOCATION => true,     // follow redirects
            CURLOPT_ENCODING       => "",       // handle all encodings
            CURLOPT_AUTOREFERER    => true,     // set referer on redirect
            CURLOPT_CONNECTTIMEOUT => 120,      // timeout on connect
            CURLOPT_TIMEOUT        => 120,      // timeout on response
            CURLOPT_MAXREDIRS      => 10,       // stop after 10 redirects
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_SSL_VERIFYPEER => 0
        );

        $ch      = curl_init( $url );
        curl_setopt_array( $ch, $options );
        $content = curl_exec( $ch );
        $err     = curl_errno( $ch );
        $errmsg  = curl_error( $ch );
        $header  = curl_getinfo( $ch );
        curl_close( $ch );

        $header['errno']   = $err;
        $header['errmsg']  = $errmsg;
        $header['content'] = $content;

        if($header['errno'] != 0) die('cURL Error - bad url, timeout, redirect loop');

        if ($header['http_code'] != 200) die('cURL Error - no page, no permissions, no service');

        return $header['content'];
    }     
    
    public function tmdbImages($id)
    {
       $movie = \App\Models\Movie::where('id',$id)->firstOrfail();

            $imdb_code = $movie->imdb_code;
            $status = 'not updated';
            try{
                $data = file_get_contents('https://api.themoviedb.org/3/movie/'.$imdb_code.'?api_key=443880ff623cdfa7c3599a4d56c167ff');
            }
            catch(\Exception $e)
            {
                return redirect()->back()->withErrors('Api Error');
            }  
            
            $data = json_decode($data);
             
            if(!empty($data->poster_path)) 
            {
                if(file_exists(ltrim($movie->cover_image,'/'))) unlink(ltrim($movie->cover_image,'/'));
                $cover_image_path = 'https://image.tmdb.org/t/p/w342'.$data->poster_path;
                $movie->cover_image = $cover_image_path;
            }
            
            if(!empty($data->backdrop_path))
            {
                if(file_exists(ltrim($movie->background_image,'/'))) unlink(ltrim($movie->background_image,'/'));
                $background_image_path = (!empty($data->backdrop_path)) ? 'https://image.tmdb.org/t/p/w1280'.$data->backdrop_path : '';
                $movie->background_image = $background_image_path;
                $status = "updated";            
            }
            
            $movie->save();
            \Cache::flush();
            
            return redirect()->back()->withSuccess($movie->title_long . " images ".$status);
              
    }
}