<?php 
namespace App\Http\Controllers\Admin;

use Backpack\CRUD\app\Http\Controllers\CrudController;

// VALIDATION: change the requests to match your own file names if you need form validation
//use App\Http\Requests\MovieCrudRequest as StoreRequest;
//use App\Http\Requests\MovieCrudRequest as UpdateRequest;

use App\Models\Genre;
use App\Models\Series;
use App\Models\Season;
use App\Models\Episode;
use App\Models\STorrent;
use App\Models\SDownload;
use App\Models\SVote;
use Illuminate\Http\Request;
use Validator;
use Config; 

class SeriesCrudController extends CrudController {

	public function setup() 
	{
        $this->crud->setModel("App\Models\Series");
        $this->crud->setRoute("admin/series");
        $this->crud->setEntityNameStrings('series', 'series');

        $this->crud->setColumns([['name'=>'id','label'=>'#','type'=>'mcheck'],'title_long','slug','findme','imdb_code','tmdb_id',['name'=>'active','type'=>'check']]);        


        $this->crud->enableAjaxTable(['title_long','genre','imdb_code','tmdb_id','year','rating','slug']);
        $this->crud->removeButton('create');
        //$this->crud->removeButton('update');
        //$this->crud->addButtonFromView('top', 'add_movie', 'add', 'end'); 
        //$this->crud->addButtonFromView('line', 'edit_movie', 'edit', 'beginning'); 
       // $this->crud->addButtonFromView('line', 'torrent', 'torrent', 'beginning'); 
        $this->crud->addButtonFromView('line', 'seasons', 'seasons', 'beginning'); 
        $this->crud->addButtonFromView('top', 'add_tmdb', 'add_tmdb', 'end'); 

        $this->crud->addButtonFromModelFunction('line', 'open_series', 'openSeries', 'beginning'); 
        $this->crud->addButtonFromModelFunction('line', 'get_torrents', 'getTorrents', 'beginning'); 
        $this->crud->addButtonFromModelFunction('line', 'get_tgx', 'getTgX', 'beginning'); 
        $this->crud->addButtonFromModelFunction('line', 'get_t1337', 'getT1337', 'beginning'); 

        $this->crud->addButtonFromView('top', 'select_all', 'select_all', 'end'); 
        $this->crud->addButtonFromView('top', 'edit_selected', 'edit_selected', 'end');  
        $this->crud->enableDetailsRow();
        $this->crud->allowAccess('details_row');

        $this->crud->addFilter([ // select2_multiple filter
          'name' => 'genres',
          'type' => 'select2',
          'label'=> 'Genres'
        ], function() { // the options that show up in the select2
            $genres = Genre::all()->pluck('name')->toArray();
            $gen = array();
            foreach($genres as $genre){
                $gen[$genre] = $genre;
            }
            return $gen;
        }, function($value) { // if the filter is active
                $this->crud->addClause('where','genres','like','%'.$value.'%');
        });  

        $this->crud->addFilter([ // select2_multiple filter
          'name' => 'genres_not',
          'type' => 'select2',
          'label'=> 'Genres Not'
        ], function() { // the options that show up in the select2
            $genres = Genre::all()->pluck('name')->toArray();
            $gen = array();
            foreach($genres as $genre){
                $gen[$genre] = $genre;
            }
            return $gen;
        }, function($value) { // if the filter is active
                $this->crud->addClause('where','genres','not like','%'.$value.'%');
        });  
        
        
        $this->crud->addFilter([ // dropdown filter
          'name' => 'torrents',
          'type' => 'dropdown',
          'label'=> 'Torrents',
          'placeholder' => 'Torrents'
        ], [
          '1' => 'Has',
          '2' => 'Has not',
        ], function($value) { // if the filter is active
            if($value == '1') $this->crud->addClause('whereHas', 'episode_torrents');
            else $this->crud->addClause('doesntHave', 'episode_torrents');
        });


        $this->crud->addFilter([ // select2_ajax filter
          'name' => 'Artist',
          'type' => 'select2_ajax',
          'label'=> 'Cast',
          'placeholder' => 'Artist name'
        ],
        url('admin/artist/ajax-artist-names'), // the ajax route
        function($value) { // if the filter is active
             $this->crud->addClause('where', 'cast','like','%'. $value .'%');
        });

        $this->crud->addFilter([ // dropdown filter
          'name' => 'background',
          'type' => 'dropdown',
          'label'=> 'Background',
          'placeholder' => 'Background Image'
        ], [
          '1' => 'Has',
          '2' => 'Has not',
        ], function($value) { // if the filter is active
            if($value == '1') $this->crud->addClause('where', 'background_image','!=','');
            else $this->crud->addClause('where', 'background_image','=',null);
        }); 

        $this->crud->addFilter([ // dropdown filter
          'name' => 'status',
          'type' => 'dropdown',
          'label'=> 'Rating'
        ], [
          '10' => '10+',
          '9' => '9+',
          '8' => '8+',
          '7' => '7+',
          '6' => '6+',
          '5' => '5+',
          '4' => '4+',
          '3' => '3+',
          '2' => '2+',
          '1' => '1+',
        ], function($value) { // if the filter is active
             $this->crud->addClause('where', 'rating','>=',$value);
        });

        $this->crud->addFilter([ // dropdown filter
          'name' => 'order_by',
          'type' => 'dropdown',
          'label'=> 'Order By'
        ], [
          'new' => 'Newly added',
          'name' => 'Name',
          'best' => 'Rating',
          'year' => 'Year',
          'download_count' => 'Download Count',
        ], function($value) { // if the filter is active
             if($value == 'new') $this->crud->orderBy('created_at','DESC');
             elseif($value == 'title_long') $this->crud->orderBy('title_long');
             elseif($value == 'best') $this->crud->orderBy('like_count','DESC');
             elseif($value == 'year') $this->crud->orderBy('year','DESC');
             elseif($value == 'download_count') $this->crud->orderBy('download_count','DESC');
             else $this->crud->orderBy('id','DESC');
        });   
        
        if(empty(request()->get('order'))) $this->crud->orderBy('created_at','DESC');


    }   

  public function create()
  {
     return view('admin.series.create');
  }

	public function store(Request $request)
	{
    $validator = Validator::make($request->all(), [
            'title' => 'required|max:100',
            'title_long' => 'required|max:120|unique:series,title_long',
            'slug' => 'nullable|max:150|unique:series,slug',
            'imdb_code' => 'required|max:120|unique:series,imdb_code',
            'year' => 'required|numeric',
            'rating' => 'required|numeric|min:1|max:10',
            'runtime' => 'numeric',
            'genres' => 'required',
            'cast' => 'max:255',
            'active' => 'required|in:0,1',
            'director' => 'max:100',
            'description_full' => '',
            'yt_trailer_code' => 'max:255',
            'language' => 'max:100',
            'mpa_rating' => 'max:100',
            'tomato_critics' => 'max:100',
            'tomato_audience' => 'max:100',
            'cover_image' => 'required|mimes:jpg,jpeg,png|max:'.Config::Get('settings.max_screenshot_size'),            
            'screenshots.*' => 'mimes:jpg,jpeg,png|max:'.Config::Get('settings.max_screenshot_size'),
            'background_image' => 'mimes:jpg,jpeg,png|max:'.Config::Get('settings.max_screenshot_size'),
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                        ->withErrors($validator)
                        ->withInput();
        }
        else {   
          $series = new Movie();
          $series->title = $request->title;
          $series->title_long = $request->title_long;
          if(!empty($request->slug)) $series->slug = $request->slug;
          $series->imdb_code = $request->imdb_code;
          $series->year = $request->year;
          $series->rating = $request->rating;
          $series->runtime = $request->runtime;
          $series->genres = implode(',', $request->genres);
          $series->cast = $request->cast;
          $series->director = $request->director;
          $series->description_full = $request->description_full;
          $series->yt_trailer_code = $request->yt_trailer_code;
          $series->language = $request->language;
          $series->mpa_rating = $request->mpa_rating;
          $series->tomato_critics = $request->tomato_critics;
          $series->tomato_audience = $request->tomato_audience;
          $series->active = $request->active;
          $series->direct_link = $request->direct_link;
          
            $destinationPath = 'images/series/'.str_slug($request->title_long);
            if (!file_exists($destinationPath)) {
            mkdir($destinationPath, 0777, true);
            chmod($destinationPath, 0777);       
          }

          $cover_image = $request->file('cover_image');
            $file_ext = $cover_image->getClientOriginalExtension();            
            $new_file_name = "cover-original.".$file_ext;
      $cover_image->move($destinationPath,$new_file_name);    
      $original_cover_image = $destinationPath."/".$new_file_name;


      $img = \Image::make($original_cover_image);
      $img->resize(230, 345);
      $resized_cover_image = $destinationPath."/"."cover-medium.".$file_ext;
      $img->save($resized_cover_image);      
      $series->cover_image = "/".$resized_cover_image;
      

      if ($request->hasFile('background_image')) {

            $background_image = $request->file('background_image');         
        $file_ext = $background_image->getClientOriginalExtension();  
        $new_file_name = "background.".$file_ext;
        $background_image->move($destinationPath,$new_file_name); 
        $series->background_image = "/".$destinationPath."/".$new_file_name;   
      }

      if($request->hasFile('screenshots')){
        $screenshots = array();
        $i = 1;
        foreach($request->file('screenshots') as $screenshot) {
          $file_ext = $screenshot->getClientOriginalExtension();  
          $new_file_name = "screenshot-".$i.".".$file_ext;
          $screenshot->move($destinationPath,$new_file_name); 
          array_push($screenshots, "/".$destinationPath."/".$new_file_name);
          $i++;                 
        }
        $series->screenshots = json_encode(array_values($screenshots));
      }


      $series->save();
      return redirect()->back()->withSuccess('Series successfully added.');
    }

	}

  public function edit($id)
  {
    $series = Series::findOrfail($id);
    return view('admin.series.edit',compact('series'));
  }

	public function update(Request $request)
	{
		 $validator = Validator::make($request->all(), [
            'id' => 'required|numeric|exists:series,id',
            'title' => 'required|max:100',
            'title_long' => 'required|max:120|unique:series,title_long,'.$request->id,
            'slug' => 'nullable|max:150|unique:series,slug,'.$request->id,
            'imdb_code' => 'required|max:120|unique:series,imdb_code,'.$request->id,
            'year' => 'required|numeric',
            'rating' => 'required|numeric|min:1|max:10',
            'runtime' => 'numeric',
            'genres' => 'required',
            'cast' => 'max:255',
            'active' => 'required|in:0,1',
            'director' => 'max:100',
            'description_full' => '',
            'yt_trailer_code' => 'max:255',
            'language' => 'max:100',
            'mpa_rating' => 'max:100',
            'tomato_critics' => 'max:100',
            'tomato_audience' => 'max:100',
            'cover_image' => 'mimes:jpg,jpeg,png|max:'.Config::Get('settings.max_screenshot_size'),            
            'screenshots.*' => 'mimes:jpg,jpeg,png|max:'.Config::Get('settings.max_screenshot_size'),
            'background_image' => 'mimes:jpg,jpeg,png|max:'.Config::Get('settings.max_screenshot_size'),
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                        ->withErrors($validator)
                        ->withInput();
        }
        else {   
          $series = Series::findOrfail($request->id);
          $series->title = $request->title;
          $series->title_long = $request->title_long;
          if(!empty($request->slug)) $series->slug = $request->slug;
          $series->imdb_code = $request->imdb_code;
          $series->year = $request->year;
          $series->rating = $request->rating;
          $series->runtime = $request->runtime;
          $series->genres = implode(',', $request->genres);
          $series->cast = $request->cast;
          $series->director = $request->director;
          $series->description_full = $request->description_full;
          $series->yt_trailer_code = $request->yt_trailer_code;
          $series->language = $request->language;
          $series->mpa_rating = $request->mpa_rating;
          $series->tomato_critics = $request->tomato_critics;
          $series->tomato_audience = $request->tomato_audience;
          $series->active = $request->active;
          $series->direct_link = $request->direct_link;
          if(!empty($request->created_at)) $series->created_at = $request->created_at;

            $destinationPath = 'images/series/'.str_slug($request->title_long);
            if (!file_exists($destinationPath)) {
        mkdir($destinationPath, 0777, true);
        chmod($destinationPath, 0777);       
      }

      if ($request->hasFile('cover_image')) {
            $cover_image = $request->file('cover_image');
              $file_ext = $cover_image->getClientOriginalExtension();            
              $new_file_name = "cover-original.".$file_ext;
        $cover_image->move($destinationPath,$new_file_name);    
        $original_cover_image = $destinationPath."/".$new_file_name;


        $img = \Image::make($original_cover_image);
        $img->resize(230, 345);
        $resized_cover_image = $destinationPath."/"."cover-medium.".$file_ext;
        $img->save($resized_cover_image);      
        $series->cover_image = "/".$resized_cover_image;
      }

      if ($request->hasFile('background_image')) {
            $background_image = $request->file('background_image');         
        $file_ext = $background_image->getClientOriginalExtension();  
        $new_file_name = "background.".$file_ext;
        $background_image->move($destinationPath,$new_file_name); 
        $series->background_image = "/".$destinationPath."/".$new_file_name;   
      }

      if($request->hasFile('screenshots')){
        $screenshots = array();
        $i = 1;
        foreach($request->file('screenshots') as $screenshot) {
          $file_ext = $screenshot->getClientOriginalExtension();  
          $new_file_name = "screenshot-".$i.".".$file_ext;
          $screenshot->move($destinationPath,$new_file_name); 
          array_push($screenshots, "/".$destinationPath."/".$new_file_name);
          $i++;                 
        }
        $series->screenshots = json_encode(array_values($screenshots));
      }


      $series->save();
      return redirect()->back()->withSuccess('Series successfully updated');
        }
	}

    public function destroy($id)
    {
      $series = Series::findOrfail($id);

      $imageFolderPath = 'images/series/'.str_slug($series->title_long);
      $torrentFolderPath = 'uploads/series/'.str_slug($series->title_long);
      if(file_exists($imageFolderPath)){        
        chmod($imageFolderPath, 0777);  
        array_map('unlink', glob($imageFolderPath.'/*'));
        rmdir($imageFolderPath);
      }      
      if(file_exists($torrentFolderPath)){
        chmod($torrentFolderPath, 0777);  
        array_map('unlink', glob($torrentFolderPath.'/*'));
        rmdir($torrentFolderPath);
      }
      Season::where('series_id',$series->id)->delete();
      Episode::where('series_id',$series->id)->delete();
      return $this->crud->delete($id);
    } 

    public function showDetailsRow($id)
    {
        $this->data['entry'] = $this->crud->getEntry($id);
        $this->data['crud'] = $this->crud;


        $seasons_missing_torrents = [];
        $episodes_missing_torrents = [];

        $seasons = \App\Models\Season::where('series_id',$id)->get(['season_number'])->pluck('season_number')->toArray();
        $season_torrents = \App\Models\STorrent::where('series_id',$id)->whereIn('season_number',$seasons)->get(['season_number'])->pluck('season_number')->toArray();
        $season_torrents = array_unique($season_torrents);
        $seasons_missing_torrents = array_diff($seasons, $season_torrents);


        foreach ($seasons as $season_number) {
          
            $episodes = \App\Models\Episode::where('series_id',$id)->where('season_number',$season_number)->get(['episode_number','slug'])->pluck('slug','episode_number')->toArray();
           
            $episode_torrents = \App\Models\ETorrent::where('series_id',$id)->where('season_number',$season_number)->whereIn('episode_number',array_keys($episodes))->get(['episode_number'])->pluck('episode_number')->toArray();
            $episode_torrents = array_unique($episode_torrents);

            $this->data['episodes'][$season_number] = $episodes; 
            $episodes_missing_torrents[$season_number] = array_diff(array_keys($episodes), $episode_torrents);

        }

        $this->data['seasons_missing_torrents'] = $seasons_missing_torrents;
        $this->data['episodes_missing_torrents'] = $episodes_missing_torrents;

        // load the view from /resources/views/vendor/backpack/crud/ if it exists, otherwise load the one in the package
        return view('crud::details_row.series', $this->data);
    } 

    public function tmdb()
    {
        return view('admin.series.tmdb')->with('title','TMDB');
    }

    public function tmdbStore(Request $request)
    {
        if(empty($request->imdb_id)) return redirect()->back()->withErrors('IMDB ID is required');
        //if(!starts_with($request->imdb_id,'tt')) return redirect()->back()->withErrors('IMDB ID is invalid');
        $id = $request->imdb_id;
        
        //$check = Series::where('imdb_code',$id)->first();
        //if(!empty($check)) return redirect()->back()->withErrors($check->title_long.' Series already exits');

        
        $with_torrents = ($request->with_torrents == 1) ? 1 : 0;
        
        $response = $this->getTmdbTv($id,$with_torrents);

        return redirect()->back()->withSuccess($response);
    }

    public function tmdbUpdateAll(Request $request)
    {
        $scrapper = \App\Models\Scrapper::where('name','tmdb_tv_update')->first();

        if ($scrapper->active != 1) {
            return 'scrapper inactive';
        }

        if (empty($scrapper->yts_ids)) return 'empty queue';
        
        $yts_ids = explode(',', $scrapper->yts_ids);

        $yts_id = array_shift($yts_ids);
        
        $scrapper->yts_ids = implode(',', $yts_ids);
        $scrapper->save();

        return $this->getTmdbTv($yts_id,1);      
    }

    public function tmdbGetTv(Request $request)
    {
        $scrapper = \App\Models\Scrapper::where('name','tmdb_get_tv_'.$request->term)->first();

        if (empty($scrapper)) {
            $scrapper = new \App\Models\Scrapper();
            $scrapper->name = 'tmdb_get_tv_'.$request->term;
            $scrapper->page = 1;
            $scrapper->limit = 100;
            $scrapper->active = 1;
            $scrapper->save();         
        }

        if ($scrapper->active != 1) {
            return 'scrapper inactive';
        }

        if (empty($scrapper->yts_ids)) 
        {            
            try{
                $query_term = '';
                if (!empty($request->term)) {
                    $query_term = '&first_air_date_year='.$request->term;
                }
                $list = file_get_contents('https://api.themoviedb.org/3/discover/tv?api_key=443880ff623cdfa7c3599a4d56c167ff&page='.$scrapper->page.$query_term);
                $list = json_decode($list);
            }
            catch(\Exception $e){
                return 'api error - '.$e->getMessage();
            }

            if(empty($list->results)){ 
                $scrapper->page = 1;
                $scrapper->save();
                return 'api error';
            }

            $yts_ids = [];
            foreach ($list->results as $list_movie) {
                array_push($yts_ids, $list_movie->id);
            }
            
            $already_exists = Series::whereIn('tmdb_id',$yts_ids)->get(['tmdb_id'])->pluck(['tmdb_id'])->toArray();
            $yts_ids = array_diff($yts_ids,$already_exists);
            
            $scrapper->page = $scrapper->page + 1;            
            $scrapper->yts_ids = implode(',', $yts_ids);
            
            if(!empty($scrapper->reset_page_from))
            {
                if($scrapper->page >= $scrapper->reset_page_from) $scrapper->page = 1;
            }            
            
            $scrapper->save();
        }   
 
        $yts_ids = explode(',', $scrapper->yts_ids);

        $yts_id = array_shift($yts_ids);
        
        $scrapper->yts_ids = implode(',', $yts_ids);
        $scrapper->save();

        $tv = Series::where('tmdb_id',$yts_id)->first(['id','title']);
        if(!empty($tv)){
            return 'already exists';
        }
        return $this->getTmdbTv($yts_id,1);      
    }
    
    public function eztvGetLatest()
    {
        $scrapper = \App\Models\Scrapper::where('name','eztv_get_latest')->first();

        if ($scrapper->active != 1) {
            return 'scrapper inactive';
        }        
        
     
        if (empty($scrapper->yts_ids)) 
        {            
            try{
              $data = file_get_contents('https://eztvx.to/api/get-torrents?limit=100&page=1');
            }
            catch(\Exception $e)
            {
              return $e->getMessage();
            }  
               
            $data = json_decode($data);

            if(empty($data->torrents)){ 
                return 'api error';
            }
            
            $yts_ids = [];
            foreach ($data->torrents as $list_movie) {
                if(!empty($list_movie->imdb_id) && strlen($list_movie->imdb_id) > 5) array_push($yts_ids, 'tt'.$list_movie->imdb_id);
            }
            
            $yts_ids = array_unique($yts_ids);
            
            //$already_exists = Series::whereIn('imdb_code',$yts_ids)->get(['imdb_code'])->pluck(['imdb_code'])->toArray();
           // $yts_ids = array_diff($yts_ids,$already_exists);

            $scrapper->yts_ids = implode(',', $yts_ids);
            
            $scrapper->save();
        }          
        
        $yts_ids = explode(',', $scrapper->yts_ids);

        $yts_id = array_shift($yts_ids);
        
        $scrapper->yts_ids = implode(',', $yts_ids);
        $scrapper->save();        
        
        $tv = Series::where('imdb_code',$yts_id)->first(['id','title']);

        return $this->getTmdbTv($yts_id,1);  
    }


    public function eztvGetAll()
    {
        $scrapper = \App\Models\Scrapper::where('name','eztv_get_all')->first();

        if ($scrapper->active != 1) {
            return 'scrapper inactive';
        }        
        
        
        if (empty($scrapper->yts_ids)) 
        {            
            try{
              $data = file_get_contents('https://eztvx.to/api/get-torrents?limit='.$scrapper->limit.'&page='.$scrapper->page);
            }
            catch(\Exception $e)
            {
              return $e->getMessage();
            }  
            
            $data = json_decode($data);
            
            if(empty($data->torrents)){ 
                return 'api error';
            }
            

            $yts_ids = [];
            foreach ($data->torrents as $list_movie) {
                if(!empty($list_movie->imdb_id) && strlen($list_movie->imdb_id) > 5) array_push($yts_ids, 'tt'.$list_movie->imdb_id);
            }
            
            $yts_ids = array_unique($yts_ids);
            
            //$already_exists = Series::whereIn('imdb_code',$yts_ids)->get(['imdb_code'])->pluck(['imdb_code'])->toArray();
           // $yts_ids = array_diff($yts_ids,$already_exists);
            
            $scrapper->page = $scrapper->page + 1;   
            $scrapper->yts_ids = implode(',', $yts_ids);
            
            if(!empty($scrapper->reset_page_from))
            {
                if($scrapper->page >= $scrapper->reset_page_from) $scrapper->page = 1;
            }
                        
            
            $scrapper->save();
        }          
        
        $yts_ids = explode(',', $scrapper->yts_ids);

        $yts_id = array_shift($yts_ids);
        
        $scrapper->yts_ids = implode(',', $yts_ids);
        $scrapper->save();        
        
        $tv = Series::where('imdb_code',$yts_id)->first(['id','title']);

        return $this->getTmdbTv($yts_id,1);  
    }



    public static function getTmdbTv($id, $with_torrents = 0)
    {
        if(starts_with($id,'tt'))
        {
          try{
              $data = file_get_contents('https://api.themoviedb.org/3/find/'.$id.'?api_key=443880ff623cdfa7c3599a4d56c167ff&external_source=imdb_id');
          }
          catch(\Exception $e)
          {
              return $e->getMessage();
          }  
          
          $data = json_decode($data);
         
          if(empty($data->tv_results[0])) return 'invalid imdb';

          $tmdb_id = $data->tv_results[0]->id;
        }
        else{
          $tmdb_id = $id;
        }
                
        try{
            $data = file_get_contents('https://api.themoviedb.org/3/tv/'.$tmdb_id.'?api_key=443880ff623cdfa7c3599a4d56c167ff&language=en-US&append_to_response=videos,images,credits,keywords,alternative_titles,external_ids');
        }
        catch(\Exception $e)
        {
            return $e->getMessage();
        }  
        
        $data = json_decode($data);
        if(!empty($data->external_ids->imdb_id)) $imdb_id = $data->external_ids->imdb_id;
        else $imdb_id = $id;

        $year = str_limit($data->first_air_date,4,'');
        $genres = [];
        
        foreach($data->genres as $g)
        {
            array_push($genres,$g->name);
        }
        
        $yt_trailer_code = (!empty($data->videos->results[0])) ? $data->videos->results[0]->key : '';
        $language = $data->original_language;
        
        $cover_image_path = 'https://image.tmdb.org/t/p/w342'.$data->poster_path;
        
        $background_image_path = (!empty($data->backdrop_path)) ? 'https://image.tmdb.org/t/p/w1280'.$data->backdrop_path : '';

        $keywords = [];

        if(!empty($data->keywords) && !empty($data->keywords->keywords))
        {
            foreach ($data->keywords->keywords as $keyword) {
                array_push($keywords, $keyword->name);
            }
        }

        $tv = Series::orWhere('tmdb_id',$data->id)->orWhere('title_long',$data->name . " (".$year.")")->first();
        if(empty($tv)){ 
          $tv = new Series();
          $tv->slug = str_slug($data->name . " (".$year.")");
          $tv->imdb_code = $imdb_id;
        }
        
        $tv->tmdb_id = $data->id;
        $tv->title = $data->name;
        $tv->title_long = $data->name . " (".$year.")";
        $tv->first_air_date = $data->first_air_date;
        $tv->last_air_date = $data->last_air_date;
        $tv->year = $year;
        $tv->genres = str_replace('&',',',implode(', ', $genres));
        $tv->keywords = implode(', ', $keywords);
        $tv->yt_trailer_code = $yt_trailer_code;
        $tv->language = $language;
        if(!empty($data->backdrop_path)) $tv->background_image = $background_image_path;
        if(!empty($data->poster_path)) $tv->cover_image = $cover_image_path;
        $tv->website = $data->homepage;
        $tv->popularity = $data->popularity;
        $tv->status = $data->status;
        $tv->tagline = $data->tagline;
        $tv->type = $data->type;
        $tv->alternative_titles = json_encode($data->alternative_titles);
        $tv->save();
        $tv->omdbUpdate();
        $tv->save();

        foreach ($data->seasons as $ds) 
        {
            try{
                $data_season = file_get_contents('https://api.themoviedb.org/3/tv/'.$tmdb_id.'/season/'.$ds->season_number.'?api_key=443880ff623cdfa7c3599a4d56c167ff&language=en-US');
            }
            catch(\Exception $e)
            {
                return $e->getMessage();
            }  

            $data_season = json_decode($data_season);

            $season = Season::where('series_id',$tv->id)->where('season_number',$data_season->season_number)->first();
            if(empty($season))
            { 
              $season = new Season();
              $season->series_id = $tv->id;
              $season->slug = str_slug($tv->title_long.' season '.$data_season->season_number);
            }
            
            $season->tmdb_id = $data_season->id;
            $season->season_number = $data_season->season_number;
            $season->air_date = $data_season->air_date;
            $season->name = $data_season->name;
            $season->description_full = $data_season->overview;
            if(!empty($data_season->poster_path)) $season->cover_image = 'https://image.tmdb.org/t/p/w342'.$data_season->poster_path;
            $season->save();



            foreach ($data_season->episodes as $data_episode) 
            {
                $episode = Episode::where('series_id',$tv->id)->where('season_number',$data_season->season_number)->where('episode_number',$data_episode->episode_number)->first();
                if(empty($episode)) 
                {
                    $episode = new Episode();
                    $episode->series_id = $tv->id;
                    $episode->season_number = $season->season_number;
                    $episode->episode_number = $data_episode->episode_number;                    
                    $episode->slug = str_slug($tv->title_long.' season '.$season->season_number.' episode '.$episode->episode_number);

                    $tv->created_at = date('Y-m-d H:i:s');
                    $tv->save();                    
                }
                $episode->tmdb_id = $data_episode->id;
                $episode->season_id = $season->id;
                $episode->name = $data_episode->name;
                $episode->air_date = $data_episode->air_date;
                $episode->overview = $data_episode->overview;
                if(!empty($data_episode->still_path)) $episode->cover_image = 'https://image.tmdb.org/t/p/w300'.$data_episode->still_path;
                $episode->save();

            }

        }

        if ($with_torrents == 1) {
           $tv->eztvUpdate();
           $tv->save();
        }

        return $tv->title_long. ' successfully added';        
    }


    public function updateSelected(Request $request)
    {
        $ids = $request->ids;
        $count = count($ids);
        if($count > 0)
        {
            $old_items = [];
            foreach($ids as $id){
              $item = Series::find($id);
              if(!empty($item))
              {
                $old_slug = $item->slug;
                  
                  
                if(!empty($request->replace))
                {
                      if(!empty($item->findme) && !empty($request->find)) $item->slug = str_replace($request->find,$request->replace,$item->slug); 
                      else $item->slug = $item->slug.'-'.$request->replace;
                      $item->findme = $request->replace;
                } 
                  
                  if(!empty($request->created_at)) $item->created_at = $request->created_at;
                  
                  if($request->featured > 0)
                  {
                      $item->featured = ($request->featured == 1) ? 1 : 0;
                  }
                  $item->save();
                  
                  
                  array_push($old_items,["series_id"=>$item->id, "slug"=>$old_slug]);
                  
              }
              
            }
            if(!empty($old_items)) \App\Models\OldLinkSeries::insert($old_items);
            return '<div class="alert alert-success">'. $count.' Items successfully updated</div>';
            
        }
        else{
            return '<div class="alert alert-danger">No Items selected</div>';
        }
    }    


    public function massEdit()
    {
        return view('admin.series.mass_edit')->with('title','Series Mass Edit');
    }
        
    public function massUpdate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'keyword' => 'required|max:100',
            'links' => 'required'
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                        ->withErrors($validator)
                        ->withInput();
        }
        $series_slugs = [];
        $season_slugs = [];
        $episode_slugs = [];

        $old_series_items = [];
        $old_season_items = [];
        $old_episode_items = [];
        foreach(preg_split("/((\r?\n)|(\r\n?))/", $request->links) as $link){
            $og_link = $link;
            $link = explode('/',$link);
            $slug = end($link);
                          
            $slug = trim($slug);
            if(strpos($og_link, '/episodes/') !== false)
            { 
              preg_match('/episode-(\d+)/', $slug,$match);

              if(!empty($match[0]))
              {
                  $pos = strpos($slug,$match[0]);
                  $slug = substr($slug,0,$pos+strlen($match[0]));
              }              
              array_push($episode_slugs,$slug);
            }
            elseif(strpos($og_link, '/seasons/') !== false)
            { 
              preg_match('/season-(\d+)/', $slug,$match);

              if(!empty($match[0]))
              {
                  $pos = strpos($slug,$match[0]);
                  $slug = substr($slug,0,$pos+strlen($match[0]));
              }                
              array_push($season_slugs, $slug);
            }
            else{ 
              $pattern = '~\b\d{4}\b\+?~';
              preg_match($pattern, $slug,$match); 
              
              if(!empty($match[0]))
              {
                  $pos = strpos($slug,$match[0]);
                  $slug = substr($slug,0,$pos+4);
              }                            
              array_push($series_slugs, $slug);
            }
        } 
        
        $series_slugs = array_unique($series_slugs);
        $season_slugs = array_unique($season_slugs);
        $episode_slugs = array_unique($episode_slugs);
        $updated = [];
 
        // Series
        foreach($series_slugs as $slug)
        {
            $item = Series::where('slug','like',$slug.'%')->first();
            if(!empty($item))
            {
                $old_slug = $item->slug;
                $item->slug = str_slug($item->title_long.' '.$request->keyword);
                $item->findme = $request->keyword;
                $item->save();
                
                array_push($old_series_items,["series_id"=>$item->id, "slug"=>$old_slug]);
                array_push($updated,$slug);
            }
        }
        if(!empty($old_series_items)) \App\Models\OldLinkSeries::insert($old_series_items);        

        // Seasons
        foreach($season_slugs as $slug)
        {
            $item = \App\Models\Season::with('series')->where('slug','like',$slug.'%')->first();
            if(!empty($item))
            {
                $old_slug = $item->slug;
                $item->slug = str_slug($item->series->title_long.' season '.$item->season_number.' '.$request->keyword);
                $item->findme = $request->keyword;
                $item->save();
                
                array_push($old_season_items,["season_id"=>$item->id, "slug"=>$old_slug]);
                array_push($updated,$slug);
            }
        }
        if(!empty($old_season_items)) \App\Models\OldLinkSeason::insert($old_season_items);        

        // Episode
        foreach($episode_slugs as $slug)
        {
            $item = \App\Models\Episode::with('series')->where('slug','like',$slug.'%')->first();
            if(!empty($item))
            {
                $old_slug = $item->slug;
                $item->slug = str_slug($item->series->title_long.' season '.$item->season_number.' episode '.$item->episode_number.' '.$request->keyword);
                $item->findme = $request->keyword;
                $item->save();
                
                array_push($old_episode_items,["episode_id"=>$item->id, "slug"=>$old_slug]);
                array_push($updated,$slug);
            }
        }
        if(!empty($old_episode_items)) \App\Models\OldLinkEpisode::insert($old_episode_items);
        
        $slugs_count = count($series_slugs) + count($season_slugs) + count($episode_slugs);
        return redirect()->back()->withSuccess(count($updated).' Items found and updated from given '.count($slugs_count).' links.');
    }     


   public function getTorrents($id)
   {
      $series = \App\Models\Series::findOrfail($id);
      return view('admin.series.get_torrents',compact('series'))->with('title','Get Torrents');
   } 

   public function storeTorrents($id,Request $request)
   {
      $series = \App\Models\Series::findOrfail($id);
      $imdb_code = $request->imdb_id;

        if(empty($imdb_code)) return redirect()->back()->withErrors('IMDB Code is required.');

        $imdb_code = str_replace('tt', '', $imdb_code);
        $page_limit = 1;
        for ($i=1; $i <= $page_limit; $i++) 
        { 
            $api = 'https://eztvx.to/api/get-torrents?limit=100&imdb_id='.$imdb_code.'&page='.$i;

            try{
                $response = file_get_contents($api);
                $data = json_decode($response);
            }
            catch(\Exception $e)
            {
                return redirect()->back()->withErrors('API Error');
            }

            $page_limit = ceil($data->torrents_count / 100);

            if(empty($data->torrents)) return redirect()->back()->withErrors('No Torrents');

            foreach ($data->torrents as $data_torrent) 
            {
                if ($data_torrent->season > 0 && $data_torrent->episode == 0) 
                {

                    $storrent = \App\Models\STorrent::where('hash',$data_torrent->hash)->where('series_id',$series->id)->where('season_number',$data_torrent->season)->first();
                    if(empty($storrent))
                    { 

                        $storrent = new \App\Models\STorrent();
                        $storrent->series_id = $series->id;
                        $storrent->season_number = $data_torrent->season;
                        $storrent->name = $data_torrent->title;
                        $storrent->hash = $data_torrent->hash;
                        $storrent->quality = $this->getQuality($data_torrent->title);
                        $storrent->resolution = $this->getResolution($storrent->quality);
                        $storrent->size_bytes = $data_torrent->size_bytes; 

                        $torrentPath = 'uploads/series/'.str_slug($series->title_long).'/'.str_slug('season '.$data_torrent->season);
                        if (!file_exists($torrentPath)) {
                            mkdir($torrentPath, 0775, true);
                            chmod($torrentPath, 0775);       
                        }                    
                        $torrent_file = $torrentPath."/".str_slug($series->title_long)."-".str_slug($storrent->quality)."-".str_random(3).".torrent";
                        try{    
                         //   file_put_contents($torrent_file, fopen($data_torrent->torrent_url, 'r'));  
                            $torrent_file = "/".$torrent_file;  
                        }   
                        catch(\Exception $e)    
                        {   
                            $torrent_file = 'magnet:?xt=urn:btih:'.$storrent->hash.'&dn='.urlencode($series->title_long).'&tr=udp://tracker.coppersurfer.tk:80&tr=udp://glotorrents.pw:6969/announce&tr=udp://tracker.leechers-paradise.org:6969&tr=udp://tracker.opentrackr.org:1337/announce&tr=udp://exodus.desync.com:6969'; 
                        } 

                        $storrent->file = $torrent_file;      

                        $series->created_at = date('Y-m-d H:i:s');             

                    }

                    $storrent->seeds = $data_torrent->seeds; 
                    $storrent->peers = $data_torrent->peers; 
                    $storrent->save();
                    continue;
                }

                $torrent = \App\Models\ETorrent::where('hash',$data_torrent->hash)->where('series_id',$series->id)->where('season_number',$data_torrent->season)->where('episode_number',$data_torrent->episode)->first();
                if(empty($torrent))
                { 
                    $torrent = new \App\Models\ETorrent();
                    $torrent->series_id = $series->id;
                    $torrent->season_number = $data_torrent->season;
                    $torrent->episode_number = $data_torrent->episode;
                    $torrent->name = $data_torrent->title;
                    $torrent->hash = $data_torrent->hash;
                    $torrent->quality = $this->getQuality($data_torrent->title);
                    $torrent->resolution = $this->getResolution($torrent->quality);
                    $torrent->size_bytes = $data_torrent->size_bytes; 

                    $torrentPath = 'uploads/series/'.str_slug($series->title_long).'/'.str_slug('season '.$data_torrent->season).'/'.str_slug('episode '.$data_torrent->episode);
                    if (!file_exists($torrentPath)) {
                        mkdir($torrentPath, 0775, true);
                        chmod($torrentPath, 0775);       
                    }                    
                    $torrent_file = $torrentPath."/".str_slug($series->title_long)."-".str_slug($torrent->quality)."-".str_random(3).".torrent";
                    try{    
                     //   file_put_contents($torrent_file, fopen($data_torrent->torrent_url, 'r'));  
                        $torrent_file = "/".$torrent_file;  
                    }   
                    catch(\Exception $e)    
                    {   
                        $torrent_file = 'magnet:?xt=urn:btih:'.$torrent->hash.'&dn='.urlencode($series->title_long).'&tr=udp://tracker.coppersurfer.tk:80&tr=udp://glotorrents.pw:6969/announce&tr=udp://tracker.leechers-paradise.org:6969&tr=udp://tracker.opentrackr.org:1337/announce&tr=udp://exodus.desync.com:6969'; 
                    } 

                    $torrent->file = $torrent_file;      

                    $series->created_at = date('Y-m-d H:i:s'); 
                    $series->save();            

                }

                $torrent->seeds = $data_torrent->seeds; 
                $torrent->peers = $data_torrent->peers; 
                $torrent->save();
                

            }
        }

        return redirect()->back()->withSuccess('Torrents Added to '.$series->title_long);
   }


    protected static function getQuality($name)
    {
        if (strpos($name, '420p') !== false) {
            $quality = '420p';
        }
        elseif (strpos($name, 'HDCAM') !== false) {
            $quality = '3D';
        }        
        elseif (strpos($name, 'HDTS') !== false) {
            $quality = 'HDTS';
        }        
        elseif (strpos($name, '720p') !== false) {
            $quality = '720p';
        }
        elseif (strpos($name, '1080p') !== false) {
            $quality = '1080p';
        }         
        elseif (strpos($name, '2160p') !== false) {
            $quality = '2160p';
        }        
        elseif (strpos($name, 'blueray') !== false) {
            $quality = 'blueray';
        }        
        elseif (strpos($name, '3D') !== false) {
            $quality = '3D';
        }        
        else{
            $quality = '720p';
        }
        return $quality;
    }     

    protected static function getResolution($torrent_quality)
    {
        if($torrent_quality == '3D'){
            $torrent_resolution = '1920*1080';
        }
        elseif($torrent_quality == '2160p'){
            $torrent_resolution = '3840*2160';
        }                            
        elseif($torrent_quality == '1080p'){
            $torrent_resolution = '1920*1080';
        }
        elseif ($torrent_quality == '720p') {
            $torrent_resolution = '1280*720';
        }
        else{
            $torrent_resolution = '1280*720';
        }
        return $torrent_resolution;
    }
    public function tgx($id)
    {
        $series = \App\Models\Series::where('id',$id)->firstOrfail();
        $seasons = \App\Models\Season::where('series_id',$id)->get(['season_number']);

        return view('admin.series.tgx',compact('series','seasons'))->with('title','TGX Scrap - '.$series->title_long);
    }    
    
    public function t1337($id)
    {
        $series = \App\Models\Series::where('id',$id)->firstOrfail();
        $seasons = \App\Models\Season::where('series_id',$id)->get(['season_number']);

        return view('admin.series.t1337',compact('series','seasons'))->with('title','1337x Scrap - '.$series->title_long);
    }
    
    

    public function curl_get_contents($url)
    {
        $user_agent='Mozilla/5.0 (Windows NT 6.1; rv:8.0) Gecko/20100101 Firefox/8.0';

        $options = array(

            CURLOPT_CUSTOMREQUEST  =>"GET",        //set request type post or get
            CURLOPT_POST           =>false,        //set to GET
            CURLOPT_USERAGENT      => $user_agent, //set user agent
            CURLOPT_COOKIEFILE     =>"cookie.txt", //set cookie file
            CURLOPT_COOKIEJAR      =>"cookie.txt", //set cookie jar
            CURLOPT_RETURNTRANSFER => true,     // return web page
            CURLOPT_HEADER         => false,    // don't return headers
            CURLOPT_FOLLOWLOCATION => true,     // follow redirects
            CURLOPT_ENCODING       => "",       // handle all encodings
            CURLOPT_AUTOREFERER    => true,     // set referer on redirect
            CURLOPT_CONNECTTIMEOUT => 120,      // timeout on connect
            CURLOPT_TIMEOUT        => 120,      // timeout on response
            CURLOPT_MAXREDIRS      => 10,       // stop after 10 redirects
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_SSL_VERIFYPEER => 0
        );

        $ch      = curl_init( $url );
        curl_setopt_array( $ch, $options );
        $content = curl_exec( $ch );
        $err     = curl_errno( $ch );
        $errmsg  = curl_error( $ch );
        $header  = curl_getinfo( $ch );
        curl_close( $ch );

        $header['errno']   = $err;
        $header['errmsg']  = $errmsg;
        $header['content'] = $content;

        if($header['errno'] != 0) die('cURL Error - bad url, timeout, redirect loop');

        if ($header['http_code'] != 200) die('cURL Error - no page, no permissions, no service');

        return $header['content'];
    }


    
}