<?php 
namespace App\Http\Controllers;

use App\User;
use App\Models\Movie;
use App\Models\Download;
use Auth;
use Validator;
use Config;
use Illuminate\Http\Request;


class UserController extends Controller 
{

	public function profile($username) 
	{
		$user = User::where('name',$username)->firstOrfail();
        $list = Download::where('user_id',$user->id)->take(6)->orderBy('created_at','desc')->get(['movie_id']);
        $movie_ids = array();
        foreach($list as $l){
            array_push($movie_ids, $l->movie_id);
        }
        $downloads = Movie::where('active',1)->whereIn('id',$movie_ids)->take(6)->get();
        $download_count = Download::where('user_id',$user->id)->count();
        $upload_count = Movie::where('user_id',$user->id)->count();
		return view('front.user.profile',compact('user','downloads','download_count','upload_count'));
	}

    public function login()
    {
        if(env('IS_DUMMY') == true) return redirect()->to(env('SOURCE_DOMAIN').'/login');
        if(Auth::check()) return redirect('/');
        return view('front.user.login')->with('page_title','Login');
    }    

    public function register()
    {
        if(env('IS_DUMMY') == true) return redirect()->to(env('SOURCE_DOMAIN').'/register');
        if(Auth::check()) return redirect('/');
        return view('front.user.register')->with('page_title','Register');
    }    

    public function recover()
    {
        if(env('IS_DUMMY') == true) return redirect()->to(env('SOURCE_DOMAIN').'/forgot-password');
        if(Auth::check()) return redirect('/');
        return view('front.user.password-reset')->with('page_title','Password Reset');
    }

	public function logout()
	{
        if(env('IS_DUMMY') == true) return redirect()->to(env('SOURCE_DOMAIN').'/logout');
		Auth::logout();
		return redirect('/');
	}

    public function edit()
    {
        $user = User::where('active',1)->where('id',Auth::user()->id)->firstOrfail();
        return view('front.user.settings',compact('user'))->with('page_title','Edit Profile');
    }

    public function update(Request $request)
    {
       $validator = Validator::make($request->all(), [
            'password' => 'sometimes|alpha_num|between:6,20|confirmed',
            'about_text' => 'max:255',
            'avatar_image' => 'sometimes|mimes:jpeg,jpg,png|max:'.Config::Get('settings.max_avatar_size'),
        ]);
        if ($validator->fails()) {
            return redirect()->back()
                        ->withErrors($validator)
                        ->withInput();
        }
        else {      
            $user = User::findOrfail(Auth::user()->id);
            if($request->password != null){ 
                $user->password = bcrypt($request->password);
            }
            $user->about_text = strip_tags($request->about_text);

            if($request->hasFile('avatar_image')){

                $destinationPath = 'images/avatars/'.str_slug($user->name);
                if (!file_exists($destinationPath)) {
                    mkdir($destinationPath, 0777, true);
                    chmod($destinationPath, 0777);       
                }                
                $avatar_image = $request->file('avatar_image');
                $file_ext = $avatar_image->getClientOriginalExtension();                  
                $new_file_name = str_random(8)."-original.".$file_ext;
                $avatar_image->move($destinationPath,$new_file_name);    
                $original_avatar_image = $destinationPath."/".$new_file_name;


                $img = \Image::make($original_avatar_image);
                $img->resize(200, 200);
                $resized_avatar_image = $destinationPath."/".str_random(8)."-medium.".$file_ext;
                $img->save($resized_avatar_image);            
                $user->avatar = "/".$resized_avatar_image;                
            }
            $user->save();
            return redirect()->back()->withSuccess('Profile successfully updated.');            
        }        
    }

    public function activate($activation_code)
    {
        $user = User::where('active',0)->where('activation_code',$activation_code)->first();
        if(!empty($user)){
            $user->active = 1;
            $user->save();
            $status = 'Account successfully activated.';
        }
        else {
            $status = 'Account activation code invalid.';
        }
        echo '<html>
                <head>
                    <meta http-equiv="refresh" content="3;url='.url('/').'" />
                </head>
                <body>
                    <h2>'.$status.'</h2>
                    <p>Redirecting in 3 seconds...</p>
                </body>
            </html>';
    }

    public function resendActivationCode($username)
    {
        $user = User::where('active',0)->where('name',$username)->first();
        if(!empty($user)){

            Mail::send('emails.activate',['user'=>$user], function ($m) use ($user) {
                $m->from(Config::Get('settings.contact_from_email'), Config::Get('settings.site_title'));
                $m->to($user->email)->subject('Activation mail');
            });
            $status = 'Account activation code successfully sent to your email address';
            echo '<html>
                <head>
                    <meta http-equiv="refresh" content="3;url='.url('/').'" />
                </head>
                <body>
                    <h2>'.$status.'</h2>
                    <p>Redirecting in 3 seconds...</p>
                </body>
            </html>';
        }
        else{
            return redirect('/');
        }
    }	
}